unit UnitICL;

interface

uses winapi.windows, system.classes, sysutils, vcl.graphics,
  vcl.controls, winapi.commctrl;

const
  E_R_InvWin16Offs = 'Offset falsch';
  E_R_InvWin16Header = 'Kein Windows 16 Header';
  E_R_NotValid16Bit = 'Keine ICL-Datei';
  E_R_NoResources = 'Keine Ressource in der Datei';
  E_R_NoNamesTable = 'Namen-Tabelle nicht gefunden';
  E_R_IconResNotFound = 'Kein Icon gefunden';
  E_R_ResTableError = 'Fehler in Ressource-Tabelle';
  E_W_NoIcons = 'Keine Icons in der Icon-Bibliothek';
  E_W_TooManyIcons = 'Mehr als 32767 Icons nicht erlaubt';
  IMAGE_DOS_SIGNATURE = $5A4D;
  IMAGE_IM_SIGNATURE = $4D49;
  IMAGE_OS2_SIGNATURE = $454E;
  IMAGE_W32_SIGNATURE = $4550;
  IMAGE_NAMETABLE_ICL: array [0 .. 3] of Char = 'ICL' + #0;

type
  PNameRec = ^TNameRec;

  TNameRec = packed record
    rnOffset: WORD;
    rnLength: WORD;
    rnFlags: WORD;
    rnID: WORD;
    rnHandle: WORD;
    rnUsage: WORD;
  end;

  PNameRecArray = ^TNamerecArray;
  TNamerecArray = array [0 .. 0] of TNameRec;

  TIconResInfo = packed record
    Width: Byte;
    Height: Byte;
    ColorCount: Byte;
    Reserved: Byte;
    Planes: WORD;
    BitCount: WORD;
    BytesInRes: DWord;
  end;

  TIconHeader = packed record
    wReserved: WORD;
    wType: WORD;
    wCount: WORD;
  end;

  TResourceIconResInfo = packed record
    ResInfo: TIconResInfo;
    ID: WORD;
  end;

  TFileIconResInfo = packed record
    ResInfo: TIconResInfo;
    dwImageOffset: DWord;
  end;

  TTypeRec = packed record
    rtTypeID: WORD;
    rtResourceCount: WORD;
    rtReserved: DWord;
  end;

  TIconImage = class
  protected
    FHandle: HIcon;
    FInfo: TIconResInfo;
    FIconData: Pointer;
    function GetHandle: HIcon; virtual;
    function GetIcon: TIcon; virtual;
    procedure CorrectIconSize;
  public
    constructor Create;
    constructor CreateCopy(Source: TIconImage); virtual;
    destructor Destroy; override;
    property Handle: HIcon read GetHandle;
    property Icon: TIcon read GetIcon;
    property Info: TIconResInfo read FInfo;
  end;

  TICLReader = class
  private
  protected
    FFilename: string;
    FIcons: TStringList;
  public
    constructor Create(Filename: string); virtual;
    destructor Destroy; override;
    function Execute: Boolean; virtual; abstract;
    property Icons: TStringList read FIcons;
  end;

  T16BitReader = class(TICLReader)
  private
    FStream: TStream;
    FAlignShift: WORD;
    FWin16Offset: WORD;
    FResourceTableStart: DWord;
    FNamestableStart: DWord;
    FIconCount: WORD;
    FGroupIconCount: WORD;
    FICLName: string;
    FIconsRes: PNameRecArray;
    FGroupIconsRes: PNameRecArray;
  protected
    procedure ReadFileHeader;
    procedure ReadResourceTable;
    procedure ReadTypeInfo;
    procedure ReadResourceNames;
  public
    constructor Create(AFilename: string); override;
    destructor Destroy; override;
    function Execute: Boolean; override;
    function GetResourceIndex(ID: WORD; GroupIcon: Boolean): Integer;
    procedure ReadResource(Index: WORD; GroupIcon: Boolean; var Data: Pointer;
      var ResName: string);
    procedure ReadResourceStream(Index: WORD; GroupIcon: Boolean;
      Stream: TStream; var ResName: string);
    property IconCount: WORD read FGroupIconCount;
  end;

  TAdvancedIcon = class;

  TIconImageList = class
  private
    FAlist: Array of Pointer;
    FCount: WORD;
    FCapacity: WORD;
    function GetIconImage(Index: WORD): TIconImage;
    procedure SetCapacity(Value: WORD);
  public
    procedure Add(IconImage: TIconImage);
    procedure Clear;
    constructor Create;
    destructor Destroy; override;
    property Count: WORD read FCount;
    property Image[Index: WORD]: TIconImage read GetIconImage;
    property Capacity: WORD read FCapacity write SetCapacity;
  end;

  TAdvancedIcon = class
  private
    FImages: TIconImageList;
  protected
    function LoadHeader(Stream: TStream; var IconCount: WORD): Boolean;
  public
    procedure Assign(Source: TAdvancedIcon); virtual;
    constructor Create; virtual;
    destructor Destroy; override;
    property Images: TIconImageList read FImages;
  end;

  TICL16Icon = class(TAdvancedIcon)
  private
    FResName: string;
  protected
    procedure LoadICL16IconImage(Reader: T16BitReader; GroupIcon: TStream);
  public
    procedure LoadICL16Icon(Reader: T16BitReader; Index: Integer);
    property ResName: string read FResName;
  end;

  TICL16IconImage = class(TIconImage)
  public
    constructor Create(AInfo: TIconResInfo; AIconData: Pointer);
  end;

  EICLReadError = class(Exception);

  THiResIcon = class(TIcon)
  private
    FHeight: Integer;
    FWidth: Integer;
    FIconChanged: Boolean;
  protected
    procedure GetIconSizeInfo; virtual;
    procedure Changed(Sender: TObject); override;
    procedure Draw(ACanvas: TCanvas; const Rect: TRect); override;
    function GetHeight: Integer; override;
    function GetWidth: Integer; override;
  public
    constructor Create; override;
  end;

function GetIconNames(Filename: string; Strings: TStrings): Boolean;
function GetIconNumber(Filename: string): Integer;
function GetAllIcons(Filename: string; ImageList: TImagelist): Integer;
function GetOneIcon(Filename: string; location: WORD): HIcon;

implementation

constructor TICLReader.Create(Filename: string);
begin
  FIcons := TStringList.Create;
  FFilename := Filename;
end;

destructor TICLReader.Destroy;
begin
  if Assigned(FIcons) then
    FIcons.free;
  inherited;
end;

procedure ICLReadError(const ErrMsg: string);
begin
  raise EICLReadError.Create(ErrMsg);
end;

procedure T16BitReader.ReadFileHeader;
var
  DosSignature: WORD;
  Win16Signature: WORD;
  ResTableOffset: WORD;
  NamesTableOffset: WORD;
begin
  with FStream do
  begin
    Read(DosSignature, SizeOf(DosSignature));
    if (DosSignature <> IMAGE_DOS_SIGNATURE) then
      ICLReadError(E_R_NotValid16Bit);
    Seek($3C, soFromBeginning);
    Read(FWin16Offset, SizeOf(WORD));
    if (FWin16Offset < $3C) then
      ICLReadError(E_R_InvWin16Offs);
    Seek(FWin16Offset, soFromBeginning);
    Read(Win16Signature, SizeOf(Win16Signature));
    if (Win16Signature <> IMAGE_OS2_SIGNATURE) then
      ICLReadError(E_R_InvWin16Header);
    Seek($22, soFromCurrent);
    Read(ResTableOffset, SizeOf(WORD));
    if (ResTableOffset = 0) then
      ICLReadError(E_R_NoResources);
    FResourceTableStart := ResTableOffset + FWin16Offset;
    Read(NamesTableOffset, SizeOf(NamesTableOffset));
    if (NamesTableOffset = 0) then
      ICLReadError(E_R_NoNamesTable);
    FNamestableStart := NamesTableOffset + FWin16Offset;
  end;
end;

procedure T16BitReader.ReadResourceTable;
var
  EndTypes: WORD;
begin
  with FStream do
  begin
    Seek(FResourceTableStart, soFromBeginning);
    Read(FAlignShift, SizeOf(WORD));
    Read(EndTypes, SizeOf(WORD));
    while (EndTypes > 0) do
    begin
      Seek(-SizeOf(WORD), soFromCurrent);
      ReadTypeInfo;
      Read(EndTypes, SizeOf(WORD));
    end;
    if (FGroupIconsRes = nil) or (FIconsRes = nil) then
      ICLReadError(E_R_ResTableError);
    ReadResourceNames;
  end;
end;

procedure T16BitReader.ReadResourceNames;
var
  NameLen: Byte;
  S: string;
begin
  with FStream do
  begin
    Read(NameLen, SizeOf(NameLen));
    while (NameLen > 0) do
    begin
      SetString(S, nil, NameLen);
      Read(Pointer(S)^, NameLen);
      if not((FIcons.Count = 0) and (CompareStr(S, StrPas(IMAGE_NAMETABLE_ICL))
        = 0)) then
        FIcons.Add(S);
      Read(NameLen, SizeOf(NameLen));
    end;
  end;
end;

procedure T16BitReader.ReadTypeInfo;
var
  TypeInfo: TTypeRec;
  I: Integer;
  TypeRead: Boolean;
begin
  with FStream do
  begin
    Read(TypeInfo, SizeOf(TypeInfo));
    with TypeInfo do
    begin
      I := SizeOf(TNameRec) * rtResourceCount;
      TypeRead := false;
      if (rtTypeID or $8000 = rtTypeID) then
      begin
        if (Lo(rtTypeID) = WORD(RT_ICON)) then
        begin
          FIconCount := rtResourceCount;
          GetMem(FIconsRes, I);
          Read(FIconsRes^, I);
          TypeRead := True;
        end;
        if (Lo(rtTypeID) = WORD(RT_GROUP_ICON)) then
        begin
          FGroupIconCount := rtResourceCount;
          GetMem(FGroupIconsRes, I);
          Read(FGroupIconsRes^, I);
          TypeRead := True;
        end;
      end;
    end;
    if not TypeRead then
      Seek(I, soFromCurrent);
  end;
end;

constructor T16BitReader.Create(AFilename: string);
begin
  inherited;
  FStream := TFileStream.Create(AFilename, fmOpenRead or fmShareDenyWrite);
  FStream.Position := 0;
  FICLName := ExtractFilename(AFilename);
  Delete(FICLName, Length(FICLName) - Length(ExtractFileExt(FICLName)) + 1, 30);
end;

destructor T16BitReader.Destroy;
begin
  FStream.free;
  inherited;
end;

function T16BitReader.Execute: Boolean;
var
  ICLIcon: TICL16Icon;
  I, L: Integer;
  S: string;
  NameOfList: Boolean;
  SL: TStringList;
begin
  try
    FIconsRes := nil;
    FGroupIconsRes := nil;
    FIcons.Clear;
    ReadFileHeader;
    ReadResourceTable;
    SL := TStringList.Create;
    try
      L := FGroupIconCount;
      NameOfList := (FIcons.Count = FGroupIconCount);
      for I := 1 to L do
      begin
        ICLIcon := TICL16Icon.Create;
        ICLIcon.LoadICL16Icon(self, I - 1); // *******
        if NameOfList then
          S := FIcons.Strings[I - 1]
        else
          S := ICLIcon.ResName;
        SL.AddObject(S, ICLIcon);
      end;
      FIcons.Assign(SL);
    finally
      SL.free;
    end;
  except
    Result := false;
    exit;
  end;
  Result := True;
end;

function T16BitReader.GetResourceIndex(ID: WORD; GroupIcon: Boolean): Integer;
var
  Res: PNameRecArray;
  I, IMax: Integer;
begin
  Result := -1;
  if GroupIcon then
  begin
    Res := FGroupIconsRes;
    IMax := FGroupIconCount;
  end
  else
  begin
    Res := FIconsRes;
    IMax := FIconCount;
  end;
  I := 0;
  while (I < IMax) do
  begin
    if (((Res^[I].rnID) xor $8000) = ID) then
    begin
      Result := I;
      break;
    end;
    INC(I);
  end;
  if Result >= 0 then
    exit;
  ICLReadError(E_R_IconResNotFound);
end;

procedure T16BitReader.ReadResource(Index: WORD; GroupIcon: Boolean;
  var Data: Pointer; var ResName: string);
var
  Namerec: TNameRec;
  ResStartPos: Longint;
  ResLength: Longint;
begin
  if GroupIcon then
    Namerec := FGroupIconsRes^[Index]
  else
    Namerec := FIconsRes^[Index];
  ResName := Format('%s %0.3d', [FICLName, Index + 1]);
  ResStartPos := Namerec.rnOffset shl FAlignShift;
  ResLength := Namerec.rnLength shl FAlignShift;
  FStream.Position := ResStartPos;
  GetMem(Data, ResLength);
  FStream.Read(Data^, ResLength);
end;

procedure T16BitReader.ReadResourceStream(Index: WORD; GroupIcon: Boolean;
  Stream: TStream; var ResName: string);
var
  Namerec: TNameRec;
  ResStartPos: Longint;
  ResLength: Longint;
begin
  if GroupIcon then
    Namerec := FGroupIconsRes^[Index]
  else
    Namerec := FIconsRes^[Index];
  ResName := Format('%s %0.3d', [FICLName, Index + 1]);
  ResStartPos := Namerec.rnOffset shl FAlignShift;
  ResLength := Namerec.rnLength shl FAlignShift;
  FStream.Position := ResStartPos;
  Stream.CopyFrom(FStream, ResLength);
end;

function TIconImageList.GetIconImage(Index: WORD): TIconImage;
begin
  Result := nil;
  if Index > FCapacity then
    exit;
  Result := FAlist[index];
end;

function TIconImage.GetHandle: HIcon;
begin
  if (FHandle = 0) then
  begin
    with Info do
      if Assigned(FIconData) then
        FHandle := CreateIconFromResourceEx(FIconData, BytesInRes, True, $30000,
          Width, Height, LR_DEFAULTCOLOR);
  end;
  Result := FHandle
end;

function TIconImage.GetIcon: TIcon;
begin
  Result := THiResIcon.Create;
  Result.Handle := GetHandle;
end;

procedure TIconImage.CorrectIconSize;
var
  X: Integer;
  BInfo: PBitMapInfoHeader;
begin
  BInfo := PBitMapInfoHeader(FIconData);
  X := BInfo^.biWidth;
  FInfo.Width := X;
  FInfo.Height := X;
end;

constructor TIconImage.Create;
begin
  FHandle := 0;
  FIconData := nil;
end;

constructor TIconImage.CreateCopy(Source: TIconImage);
begin
  Create;
  FInfo := Source.Info;
  GetMem(FIconData, FInfo.BytesInRes);
  if Assigned(Source.FIconData) then
    system.Move(Source.FIconData^, FIconData^, FInfo.BytesInRes);
  if (Source.Handle <> 0) then
    FHandle := CopyIcon(Source.Handle);
  CorrectIconSize;
end;

destructor TIconImage.Destroy;
begin
  DestroyIcon(FHandle);
  if Assigned(FIconData) then
  begin
    Freemem(FIconData, Info.BytesInRes);
  end;
end;

procedure TIconImageList.SetCapacity(Value: WORD);
begin
  if Value < FCount then
    Value := FCount;
  setlength(FAlist, Value * SizeOf(Pointer));
  FCapacity := Value;
end;

procedure TIconImageList.Add(IconImage: TIconImage);
begin
  INC(FCount);
  if FCount > FCapacity then
    SetCapacity(FCount);
  FAlist[FCount - 1] := IconImage;
end;

procedure TIconImageList.Clear;
var
  I: Integer;
begin
  for I := 1 to FCount do
    TIconImage(FAlist[I - 1]).free;
  FAlist := nil;
  FCapacity := 0;
  FCount := 0;
end;

constructor TIconImageList.Create;
begin
  FAlist := nil;
  FCount := 0;
  FCapacity := 0;
end;

destructor TIconImageList.Destroy;
begin
  Clear;
  inherited;
end;

function TAdvancedIcon.LoadHeader(Stream: TStream; var IconCount: WORD)
  : Boolean;
var
  Header: TIconHeader;
begin
  Result := false;
  with Stream do
  begin
    Read(Header, SizeOf(Header));
    if not((Header.wReserved = 0) and (Header.wType = 1)) then
      exit;
    IconCount := Header.wCount;
    Images.Capacity := IconCount;
    Result := True;
  end;
end;

procedure TAdvancedIcon.Assign(Source: TAdvancedIcon);
var
  I: Integer;
  SourceImage, DestImage: TIconImage;
begin
  Images.Clear;
  Images.Capacity := Source.Images.Count;
  for I := 1 to Source.Images.Count do
  begin
    SourceImage := Source.Images.Image[I - 1];
    if Assigned(SourceImage) then
    begin
      DestImage := TIconImage.CreateCopy(SourceImage);
      Images.Add(DestImage);
    end;
  end;
end;

constructor TAdvancedIcon.Create;
begin
  FImages := TIconImageList.Create;
end;

destructor TAdvancedIcon.Destroy;
begin
  FImages.free;
end;

procedure TICL16Icon.LoadICL16IconImage(Reader: T16BitReader;
  GroupIcon: TStream);
var
  Info: TResourceIconResInfo;
  Image: TICL16IconImage;
  Index: Integer;
  S: string;
  Data: Pointer;
begin
  GroupIcon.Read(Info, SizeOf(Info));
  Index := Reader.GetResourceIndex(Info.ID, false);
  if Index < 0 then
    exit;
  Reader.ReadResource(Index, false, Data, S);
  Image := TICL16IconImage.Create(Info.ResInfo, Data);
  Images.Add(Image);
end;

procedure TICL16Icon.LoadICL16Icon(Reader: T16BitReader; Index: Integer);
var
  GroupIcon: TMemoryStream;
  IconCount: WORD;
  I: Integer;
begin
  GroupIcon := TMemoryStream.Create;
  try
    Reader.ReadResourceStream(Index, True, GroupIcon, FResName);
    GroupIcon.Position := 0;
    if not LoadHeader(GroupIcon, IconCount) then
      exit;
    for I := 1 to IconCount do
      LoadICL16IconImage(Reader, GroupIcon);
  finally
    GroupIcon.free;
  end;
end;

constructor TICL16IconImage.Create(AInfo: TIconResInfo; AIconData: Pointer);
begin
  inherited Create;
  FInfo := AInfo;
  FIconData := AIconData;
end;

procedure THiResIcon.GetIconSizeInfo;
var
  IconInfo: TIconInfo;
  Bitmap: winapi.windows.TBitmap;
  Res: Integer;
begin
  if not FIconChanged then
    exit;
  FIconChanged := false;
  if not GetIconInfo(Handle, IconInfo) then
    exit;
  Res := GetObject(IconInfo.hbmColor, SizeOf(Bitmap), @Bitmap);
  if Res = 0 then
    Res := GetObject(IconInfo.hbmMask, SizeOf(Bitmap), @Bitmap);
  if Res = 0 then
    exit;
  DeleteObject(IconInfo.hbmColor);
  DeleteObject(IconInfo.hbmMask);
  FWidth := Bitmap.bmWidth;
  FHeight := Bitmap.bmWidth;
end;

procedure THiResIcon.Changed(Sender: TObject);
begin
  inherited;
  FIconChanged := True;
end;

procedure THiResIcon.Draw(ACanvas: TCanvas; const Rect: TRect);
var
  DC: HDC;
begin
  with Rect.TopLeft do
  begin
    GetIconSizeInfo;
    DC := ACanvas.Handle;
    if (DC <> 0) then
    begin
      DrawIconEx(DC, X, Y, Handle, Width, Height, 0, 0, DI_NORMAL);
    end;
  end;
end;

function THiResIcon.GetHeight: Integer;
begin
  GetIconSizeInfo;
  Result := FHeight;
end;

function THiResIcon.GetWidth: Integer;
begin
  GetIconSizeInfo;
  Result := FWidth;
end;

constructor THiResIcon.Create;
begin
  inherited;
  FHeight := GetSystemMetrics(SM_CYICON);
  FWidth := GetSystemMetrics(SM_CXICON);
  FIconChanged := false;
end;

function GetIconNames(Filename: string; Strings: TStrings): Boolean;
var
  Reader: TICLReader;
begin
  Strings.Clear;
  Result := false;
  Reader := T16BitReader.Create(Filename);
  try
    if Reader.Execute then
    begin
      Strings.Assign(Reader.Icons);
      Result := True;
    end;
  except
  end;
  Reader.free;
end;

function GetIconNumber(Filename: string): Integer;
var
  Reader: TICLReader;
begin
  Result := 0;
  Reader := T16BitReader.Create(Filename);
  try
    if Reader.Execute then
      Result := Reader.Icons.Count;
  except
  end;
  Reader.free;
end;

function GetAllIcons(Filename: string; ImageList: TImagelist): Integer;
var
  iclList: TStringList;
  ICLIcon: TAdvancedIcon;
  X: Integer;
begin
  Result := 0;
  ImageList.Clear;
  ImageList.Width := 32;
  ImageList.Height := 32;
  iclList := TStringList.Create;
  if GetIconNames(Filename, iclList) then
  begin
    try
      for X := 0 to iclList.Count - 1 do
      begin
        ICLIcon := TAdvancedIcon(iclList.Objects[X]);
        ImageList_AddIcon(ImageList.Handle, ICLIcon.Images.Image[0].Handle);
      end;
      Result := iclList.Count;
    except
    end;
  end;
  iclList.free;
end;

function GetOneIcon(Filename: string; location: WORD): HIcon;
var
  iclList: TStringList;
  ICLIcon: TAdvancedIcon;
begin
  Result := 0;
  iclList := TStringList.Create;
  if GetIconNames(Filename, iclList) then
  begin
    try
      ICLIcon := TAdvancedIcon(iclList.Objects[location]);
      Result := ICLIcon.Images.Image[0].Handle;
    except
    end;
  end;
  iclList.free;
end;

end.
